document.addEventListener('DOMContentLoaded', () => {
    const addDominioBtn = document.getElementById('addDominioBtn');
    const dominiosContainer = document.getElementById('dominiosContainer');
    const cotizacionForm = document.getElementById('cotizacionForm');

    // Set today's date as the default value for the cotizacion date
    const today = new Date().toISOString().split('T')[0];
    document.getElementById('fechaCotizacion').value = today;

    addDominioBtn.addEventListener('click', () => {
        dominiosContainer.style.display = 'block';
        addNewDominio();
    });

    cotizacionForm.addEventListener('input', updateTotals);

    cotizacionForm.addEventListener('submit', (event) => {
        event.preventDefault();

        const formData = new FormData(event.target);
        fetch('registrar_cotizacion.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                window.open('generar_pdf.php?id_cotizacion=' + data.id_cotizacion, '_blank');
                window.location.reload();
            } else {
                alert('Error al registrar la cotización');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error al registrar la cotización');
        });
    });

    function addNewDominio() {
        const newDominioIndex = document.querySelectorAll('.dominio').length;
        const newDominio = document.createElement('div');
        newDominio.classList.add('dominio', 'mb-4', 'p-3', 'border', 'rounded');
        newDominio.innerHTML = `
            <div class="form-row">
                <div class="form-group col-md-6">
                    <label for="dominio">Dominio:</label>
                    <input type="text" class="form-control" name="dominio[]" required>
                </div>
                <div class="form-group col-md-6">
                    <label for="jurisdiccion">Jurisdicción:</label>
                    <select class="form-control jurisdiccion" name="jurisdiccion[${newDominioIndex}]" required>
                        <!-- Opciones de jurisdicción cargadas dinámicamente -->
                    </select>
                </div>
            </div>
            <div class="table-responsive">
                <table class="table table-bordered">
                    <thead>
                        <tr>
                            <th>Nro de Acta</th>
                            <th>Importe</th>
                            <th>% de Costo</th>
                            <th>% de Descuento</th>
                            <th>Observaciones</th>
                        </tr>
                    </thead>
                    <tbody class="infraccionesContainer">
                        <tr class="infraccion">
                            <td><input type="text" class="form-control" name="nroActa[${newDominioIndex}][]" required style="min-width: 100px;"></td>
                            <td><input type="number" class="form-control importe" step="1" name="importe[${newDominioIndex}][]" required style="min-width: 100px;"></td>
                            <td><input type="number" class="form-control porcentajeCosto" step="1" name="porcentajeCosto[${newDominioIndex}][]" required style="min-width: 100px;"></td>
                            <td><input type="number" class="form-control porcentajeDescuento" step="1" name="porcentajeDescuento[${newDominioIndex}][]" value="50" required style="min-width: 100px;"></td>
                            <td><input type="text" class="form-control" name="observaciones[${newDominioIndex}][]" style="min-width: 100px;"></td>
                        </tr>
                    </tbody>
                </table>
            </div>
            <button type="button" class="btn btn-secondary addInfraccionBtn">Agregar Infracción</button>
        `;
        dominiosContainer.appendChild(newDominio);

        newDominio.querySelector('.addInfraccionBtn').addEventListener('click', () => {
            addNewInfraccion(newDominioIndex, newDominio);
        });

        loadJurisdicciones(newDominio).then(() => {
            setDefaultJurisdiccion(newDominioIndex, newDominio);
            updateTotals();
            addDescuentoValidation(newDominio);
        });

        newDominio.querySelector('.jurisdiccion').addEventListener('change', function() {
            updateJurisdiccionCosts(newDominioIndex, newDominio);
        });
    }

    function addNewInfraccion(dominioIndex, dominioElement) {
        const porcentajeCostoActual = dominioElement.querySelector('.jurisdiccion').selectedOptions[0].dataset.porcentajeCosto;

        const newInfraccion = document.createElement('tr');
        newInfraccion.classList.add('infraccion');
        newInfraccion.innerHTML = `
            <td><input type="text" class="form-control" name="nroActa[${dominioIndex}][]" required style="min-width: 100px;"></td>
            <td><input type="number" class="form-control importe" step="1" name="importe[${dominioIndex}][]" required style="min-width: 100px;"></td>
            <td><input type="number" class="form-control porcentajeCosto" step="1" name="porcentajeCosto[${dominioIndex}][]" value="${porcentajeCostoActual}" required style="min-width: 100px;"></td>
            <td><input type="number" class="form-control porcentajeDescuento" step="1" name="porcentajeDescuento[${dominioIndex}][]" value="50" required style="min-width: 100px;"></td>
            <td><input type="text" class="form-control" name="observaciones[${dominioIndex}][]" style="min-width: 100px;"></td>
        `;
        dominioElement.querySelector('.infraccionesContainer').appendChild(newInfraccion);
        updateTotals();
        addDescuentoValidation(newInfraccion);
    }

    function loadJurisdicciones(element) {
        return fetch('obtener_jurisdicciones.php')
        .then(response => response.json())
        .then(data => {
            const jurisdicciones = data.jurisdicciones;
            const jurisdiccionSelects = element.querySelectorAll('.jurisdiccion');
            jurisdiccionSelects.forEach(select => {
                jurisdicciones.forEach(jurisdiccion => {
                    const option = document.createElement('option');
                    option.value = jurisdiccion.id;
                    option.textContent = jurisdiccion.nombre;
                    option.dataset.porcentajeCosto = jurisdiccion.porcentaje_costo;
                    select.appendChild(option);
                });

                // Set default percentage cost when loading jurisdictions
                setDefaultJurisdiccion(select.closest('.dominio').dataset.index, select.closest('.dominio'));
            });
        })
        .catch(error => console.error('Error al cargar las jurisdicciones:', error));
    }

    function setDefaultJurisdiccion(dominioIndex, dominioElement) {
        const firstOption = dominioElement.querySelector('.jurisdiccion').options[0];
        const porcentajeCosto = firstOption.dataset.porcentajeCosto;

        const infracciones = dominioElement.querySelectorAll('.infraccion');
        infracciones.forEach(infraccion => {
            infraccion.querySelector('.porcentajeCosto').value = porcentajeCosto;
        });
    }

    function updateJurisdiccionCosts(dominioIndex, dominioElement) {
        const selectedJurisdiccion = dominioElement.querySelector('.jurisdiccion').value;
        const selectedOption = dominioElement.querySelector(`.jurisdiccion option[value="${selectedJurisdiccion}"]`);
        const porcentajeCosto = selectedOption.dataset.porcentajeCosto;

        const infracciones = dominioElement.querySelectorAll('.infraccion');
        infracciones.forEach(infraccion => {
            infraccion.querySelector('.porcentajeCosto').value = porcentajeCosto;
        });

        updateTotals();
    }

    function updateTotals() {
        let totalAdeudado = 0;
        let totalCosto = 0;
        let totalAbonar = 0;

        document.querySelectorAll('.table .infraccion').forEach(row => {
            const importe = parseFloat(row.querySelector('.importe').value) || 0;
            const porcentajeCosto = parseFloat(row.querySelector('.porcentajeCosto').value) || 0;
            const porcentajeDescuento = parseFloat(row.querySelector('.porcentajeDescuento').value) || 0;

            totalAdeudado += importe;
            totalCosto += importe * (porcentajeCosto / 100);
            totalAbonar += importe - (importe * (porcentajeDescuento / 100));
        });

        document.getElementById('totalAdeudado').textContent = totalAdeudado.toFixed(2);
        document.getElementById('totalCosto').textContent = totalCosto.toFixed(2);
        document.getElementById('totalAbonar').textContent = totalAbonar.toFixed(2);
        document.getElementById('totalGanancia').textContent = (totalAbonar - totalCosto).toFixed(2);
    }

    function addDescuentoValidation(element) {
        const descuentoInputs = element.querySelectorAll('.porcentajeDescuento');
        descuentoInputs.forEach(descuentoInput => {
            descuentoInput.addEventListener('blur', function() {
                const row = this.closest('.infraccion');
                const porcentajeCosto = parseFloat(row.querySelector('.porcentajeCosto').value) || 0;
                const porcentajeDescuento = parseFloat(this.value) || 0;
                if (porcentajeCosto + porcentajeDescuento > 100) {
                    alert('No se puede hacer un % de descuento mayor al costo');
                    this.value = '';
                }
                updateTotals();
            });
        });
    }

    // Load jurisdictions and add validation for the initial domain
    const initialDomain = document.querySelector('.dominio');
    loadJurisdicciones(initialDomain).then(() => {
        setDefaultJurisdiccion(0, initialDomain);
        updateTotals();
        addDescuentoValidation(initialDomain);
    });
});
